;;; kbm-cheat-sheet.el --- キーボードマクロ操作チートシートを表示する -*- lexical-binding: nil -*-
;;;
;;; Version: 2025/12/29
;;;
;;; Install
;;;  copy mcalendar.el to /usr/share/emacs/site-lisp/.
;;;
;;; Setting
;;;  add to .emacs
;;;  ----
;;;  (require 'kbm-cheat-sheet)
;;;  ----
;;;
;;; Start
;;;  M-x kbm-cheat-sheet
;;;   or
;;;  M-x kbm-cheat-sheet-other-window
;;;

;;; Code:

(defcustom kbm-cheat-sheet-truncate-lines t "" :group 'kbm-cheat-sheet :type 'boolean)
(defcustom kbm-cheat-sheet-text-scale  -2.0 "" :group 'kbm-cheat-sheet :type 'float)
(defcustom kbm-cheat-sheet-dsp-colmuns   60 "" :group 'kbm-cheat-sheet :type 'integer)

(defvar kbm-cheat-sheet-buffername "*kbm-cheat-sheet*")
(defvar kbm-cheat-sheet-local-map nil)

(defun kbm-cheat-sheet (&optional othwin)
  "Display key-board-macro cheat sheet"
  (interactive)

  (let ((selected))
    (setq selected (selected-window))
    (if othwin
        (switch-to-buffer-other-window kbm-cheat-sheet-buffername)
      (switch-to-buffer kbm-cheat-sheet-buffername))
    (kbm-cheat-sheet-print)
    (setq kbm-cheat-sheet-local-map (make-keymap))
    (define-key kbm-cheat-sheet-local-map "q"  'kbm-cheat-sheet-quit)
    (use-local-map kbm-cheat-sheet-local-map)
    (text-scale-set kbm-cheat-sheet-text-scale)
    (select-window selected))
  )

(defun kbm-cheat-sheet-other-window (&optional othwin)
  (interactive)
  (kbm-cheat-sheet t))

(defun kbm-cheat-sheet-quit ()
  (interactive)
  (kill-buffer kbm-cheat-sheet-buffername))

(defun kbm-cheat-sheet-print ()
  (setq truncate-lines kbm-cheat-sheet-truncate-lines)
  (setq cursor-type t)
  (read-only-mode 0)
  (buffer-disable-undo)
  (erase-buffer)

  (insert (format "%s" (propertize
                        "キーボードマクロ操作チートシート"
                        'face '(:underline t :height 1.2))))
  (insert "\n\n")

  (insert (format "%s" (propertize
                        "基本のキーバインド"
                        'face '(:underline t))))
  (insert "\n\n")
  (insert (kbm-cheat-sheet-print-basicbinds))
  (insert "\n")

  (insert (format "%s" (propertize
                        "キーバインドされたマクロ一覧"
                        'face '(:underline t))))
  (insert "\n\n")
  (insert (kbm-cheat-sheet-print-binded-macros))
  (insert "\n")

  (insert (format "%s" (propertize
                        "キーボードマクロリング"
                        'face '(:underline t))))
  (insert "\n\n")
  (insert (kbm-cheat-sheet-print-kbm-ring))

  (read-only-mode 1)
  (goto-char (point-min))
  )

(defun kbm-cheat-sheet-print-basicbinds ()
  (let ((str ""))

    (setq str
          (concat
           (format "  %-12s : %s(current=%s); %s\n"
                   "C-x C-k C-c"
                   "キーボードマクロカウンターをセットします"
                   (format kmacro-counter-format kmacro-last-counter)
                   "(kmacro-set-counter)")
           (format "  %-12s : %s; %s\n"
                   "C-x C-k C-a"
                   "引数をキーボードマクロカウンタ値に加えます"
                   "(kmacro-add-counter)")
           (format "  %-12s : %s(current=\"%s\"); %s\n"
                   "C-x C-k C-f"
                   "キーボードマクロカウンタの書式を指定します"
                   kmacro-counter-format
                   "(kmacro-set-format)")
           (format "  %-12s : %s; %s\n"
                   "C-x C-k b"
                   "一番最後に定義したキーボードマクロをキー([0-9A-Z])にバインドします"
                   "(kmacro-bind-to-key)")
           (format "  %-12s : %s; %s\n"
                   "C-x C-k C-p"
                   "キーボードマクロリングを前のマクロにローテートします"
                   "(kmacro-cycle-ring-previous)")
           (format "  %-12s : %s; %s\n"
                   "C-x C-k C-n"
                   "キーボードマクロリングを次のマクロにローテートします"
                   "(kmacro-cycle-ring-next)")
           (format "  %-12s : %s;\n"
                   "C-u C-u F3"
                   "最後のキーボードマクロを再実行せずにキーをマクロ定義に追加します")
           (format "  %-12s : %s; %s\n"
                   "C-x C-k r"
                   "リージョンの各行に対して最後のキーボードマクロを実行します"
                   "(apply-macro-to-region-lines)")
           ))
    str
    ))

(defun kbm-cheat-sheet-print-binded-macros ()
  (let ((kbm-seq-bind-list ""))
    (when (version< emacs-version "28.0")
      (dolist (elm (cdr kmacro-keymap))
        (let ((macro-str))
          (if (or (and (>= (car elm) ?0)
                       (<= (car elm) ?9))
                  (and (>= (car elm) ?A)
                       (<= (car elm) ?Z)))
              (if (string= (format "%s" (car (cdr elm))) "lambda")
                  (progn
                    (setq macro-str
                          (format "%s" (kbm-cheat-sheet-kmacro-display
                                        (car (car (cdr (car (cdr (nth 4 (cdr elm)))))))
                                        kbm-cheat-sheet-dsp-colmuns "Seq")))
                    (setq kbm-seq-bind-list
                          (concat
                           kbm-seq-bind-list
                           (format "  %-12s : %s\n"
                                   (format "C-x C-k %c" (car elm))
                                   macro-str)))
                    ))
            )))
      )
    (when (and (version<= "28.0" emacs-version)
               (version<  emacs-version "29.0"))
      (dolist (elm (cdr kmacro-keymap))
        (let ((macro-str))
          (if (or (and (>= (car elm) ?0)
                       (<= (car elm) ?9))
                  (and (>= (car elm) ?A)
                       (<= (car elm) ?Z)))
              (if (string= (format "%s" (aref (aref (cdr (car (cdr kmacro-keymap))) 2) 1))
                           "kmacro--extract-lambda")
                  (progn
                    (setq macro-str
                          (format "%s" (kbm-cheat-sheet-kmacro-display
                                        (car (aref (aref (cdr elm) 2) 0))
                                        kbm-cheat-sheet-dsp-colmuns "Seq")))
                    (setq kbm-seq-bind-list
                          (concat
                           kbm-seq-bind-list
                           (format "  %-12s : %s\n"
                                   (format "C-x C-k %c" (car elm))
                                   macro-str)))
                    ))
            )))
      )
    (when (version<= "29.0" emacs-version)
      (dolist (elm (cdr kmacro-keymap))
        (let ((macro-str))
          (if (or (and (>= (car elm) ?0)
                       (<= (car elm) ?9))
                  (and (>= (car elm) ?A)
                       (<= (car elm) ?Z)))
              (progn
                (setq macro-str
                      (format "%s" (kbm-cheat-sheet-kmacro-display
                                    (aref (aref (cdr elm) 2) 0)
                                    kbm-cheat-sheet-dsp-colmuns "Seq")))
                (setq kbm-seq-bind-list
                      (concat
                       kbm-seq-bind-list
                       (format "  %-12s : %s\n"
                               (format "C-x C-k %c" (car elm))
                               macro-str)))
                )
            )))
      )
    kbm-seq-bind-list
  ))

(defun kbm-cheat-sheet-print-kbm-ring ()
  (let ((list-str ""))
    (if last-kbd-macro
        (progn
          (when (version< emacs-version "29.0")
            (setq list-str (format "  > %s\n"
                                   (kbm-cheat-sheet-kmacro-display
                                    last-kbd-macro kbm-cheat-sheet-dsp-colmuns "Seq")))
            (unless (null kmacro-ring)
              (dolist (elm kmacro-ring)
                (setq list-str
                      (concat
                       list-str
                       (format "    %s\n"
                               (kbm-cheat-sheet-kmacro-display
                                (car elm) kbm-cheat-sheet-dsp-colmuns "Seq"))))
                )))
          (when (version<= "29.0" emacs-version)
            (setq list-str (format "  > %s\n"
                                   (kbm-cheat-sheet-kmacro-display
                                    last-kbd-macro kbm-cheat-sheet-dsp-colmuns "Seq")))
            (unless (null kmacro-ring)
              (dolist (elm kmacro-ring)
                (setq list-str
                      (concat
                       list-str
                       (format "    %s\n"
                               (kbm-cheat-sheet-kmacro-display
                                (aref (aref elm 2) 0) kbm-cheat-sheet-dsp-colmuns "Seq"))))
                ))))
      )
    list-str
    ))


(defun kbm-cheat-sheet-kmacro-display (macro &optional trunc descr empty)
  "Display a keyboard MACRO.
Optional arg TRUNC specifies the macro display width.
Optional arg DESCR is descriptive text for macro; default is \"Macro:\".
Optional arg EMPTY is message to print if no macros are defined."
  (if macro
      (let* ((x 60)
             (m (format-kbd-macro macro 1))
             (l (length m))
             (z (or (if (and (integerp trunc) (>= trunc 0)) trunc nil) (if (> l x) x l)))
             (s (substring m 0 (if (= z 0) nil (if (< l z) l z)))))
        (format "%s%s: %s%s" (or descr "Macro")
                 (if (= kmacro-counter 0) ""
                   (format " [%s]"
                           (format kmacro-counter-format-start kmacro-counter)))
                 s (if (< (length s) l) "..." "")
                 ))
    (format "%s" (or empty "No keyboard macros defined"))))

(provide 'kbm-cheat-sheet)
