/**
  概要
    MinGWビルドの EmacsでImageMagickの magick convertコマンドを使用するケースに
    代用する事で magick convertコマンドそのままでいくつか起こる問題を解消したり
    機能拡張します。

 使い方
   0. 本プログラムはは ImageMagickの magick convertコマンドを使用します。
      もしインストールされていない場合はインストールしてください。
      MinGWでは pacmanでパッケージインストールできます。
      デフォルトではパスの通った magick convertコマンドを使用します。

   1. 本プログラムはD言語で書かれています。
      dlang.org(https://dlang.org/)からコンパイラ(dmd)をダウンロード＆インストール
      しておき、本ソースコードを
       $  dmd -O -release wrapped_convert.d
      のようにコンパイルすると wrapped_convert.exeが生成されます。
      パスの通ったディレクトリに配置するなどして Emacsから実行できるようにして
      ください。

   2. JXRのデコーダー(JxrDecApp) および BPGのデコーダー(bpgdec) および
      JXLのデコーダー(djxl)を予めインストールしておく必要があります。
      JXRとJXLはpacmanでパッケージインストールできます。
      BPGは公式(https://bellard.org/bpg/)からバイナリをダウンロードできます。
      いずれもソースコードからビルドしても構わないと思います。

   3. .emacs に以下の設定を追加してください。
      1.でパスの通ったディレクトリに実行ファイルをインストールした場合の例示です。
      ご自身の設定とぶつかる場合は都合に合わせて変更してください。

      ;;-----------------------------------------------------------------
      ;; 組み込みImageMagickでの表示除外リスト
      ;;-----------------------------------------------------------------
      (when (image-type-available-p 'imagemagick)
        (dolist (imgtype (list 'GIF 'GIF87 'JPEG 'JPG 'PNG 'PNG24 'PNG32 'PNG8
                               'PNM 'PBM 'PGM 'PPM 'SVG 'SVGZ 'TIFF 'TIFF64
                               'TIF 'XBM 'XPM))
          (add-to-list 'imagemagick-types-inhibit imgtype))
        (add-to-list 'imagemagick-types-inhibit 'JXR)
        (add-to-list 'imagemagick-types-inhibit 'BPG)
        (add-to-list 'imagemagick-types-inhibit 'JXL)
        (imagemagick-register-types)
        )

      (add-to-list 'image-format-suffixes '(image/tga "tga"))

      (modify-coding-system-alist 'process "convert" 'cp932)
      (modify-coding-system-alist 'process "wrapped_convert" 'cp932)
      ;;-----------------------------------------------------------------
      ;; image-dired setting
      ;;-----------------------------------------------------------------
      (dolist (imgext (list "xcf" "psd" "png" "jpeg" "jpg" "jfif" "gif" "tiff" "tif"
                            "xbm" "xpm" "pbm" "pgm" "ppm" "svg" "bmp" "jp2" "heic"
                            "jxr" "jxl" "bpg" "webp" "tga"))
        (add-to-list 'image-file-name-extensions imgext))
      (setq image-dired-cmd-create-thumbnail-program
            "wrapped_convert")
      (setq image-dired-cmd-create-thumbnail-options
            '("-auto-orient" "%f[0]" "-size" "%wx%h" "-resize" "%wx%h>"
              "-background" "white" "-gravity" "center" "-extent" "%wx%h"
              "-strip" "jpeg:%t"))
      (setq image-dired-cmd-create-temp-image-program
            "wrapped_convert")
      (setq image-dired-cmd-create-temp-image-options
            '("-auto-orient" "%f[0]" "-size" "%wx%h" "-resize" "%wx%h>" "-strip"
              "jpeg:%t"))
      ;;-----------------------------------------------------------------
      ;; image-mode setting
      ;;-----------------------------------------------------------------
      (setq image-type-file-name-regexps
            '(("\\.png\\'" . png)
              ("\\.gif\\'" . gif)
              ("\\.jpe?g\\'" . jpeg)
      ;;        ("\\.bmp\\'" . bmp)  ;workaround for failed to display BMP file.
              ("\\.xpm\\'" . xpm)
              ("\\.pbm\\'" . pbm)
              ("\\.xbm\\'" . xbm)
              ("\\.ps\\'" . postscript)
              ("\\.tiff?\\'" . tiff)
              ("\\.svgz?\\'" . svg)
              ))
      (setq image-converter--converters
            '((graphicsmagick :command ("gm" "convert") :probe ("-list" "format"))
              (ffmpeg :command "ffmpeg" :probe "-decoders")
              (imagemagick :command "wrapped_convert" :probe ("-list" "format"))))
      (add-to-list 'auto-mode-alist '("\\.jp2$"  . image-mode))
      (add-to-list 'auto-mode-alist '("\\.heic$" . image-mode))
      (add-to-list 'auto-mode-alist '("\\.jxr$"  . image-mode))
      (add-to-list 'auto-mode-alist '("\\.bpg$"  . image-mode))
      (add-to-list 'auto-mode-alist '("\\.jxl$"  . image-mode))
      (setq image-converter 'imagemagick)
      (setq image-use-external-converter t)
      ;-------

 解消が期待される問題
   - 外部 magick convertコマンドを介して表示する際、JXR,BPG フォーマットの
     画像が表示対象にならないのを解消します。
     JXRのデコーダー(JxrDecApp) および BPGのデコーダー(bpgdec)がインストール
     されていていれば magick convertコマンドとしてはデコード可能なハズですが、
     MinGWの magick convertはうまく動作しないようです。
     また、「magick convert -list format」にも対応フォーマットとして表示されない
     ため、Emacsはデコードできないフォーマットと判断してしまいます。
     本プログラムでは JXR,BPG のデコーダーの存在有無に従って
     magick convert -list format にJXR,BPGの情報を追加します。

   - 画像よってデコードはできるが標準エラー出力(stderr)にメッセージが出ると
     Emacsでの画像表示に失敗する事があるのを解消します。
     本スクリプトで標準エラー出力を完全に捨てる事で対応しています。

   - Emacsでのリモートファイル表現(/ssh:user@host:/pathes/filename.ext など)
     を解釈するようになります。
     sshを使用してファイル転送を行なう事で、image-dired でリモートの画像ファイルの
     サムネイル表示や画像表示が可能になります。
     予めパスワード無しでログインできるように公開鍵認証設定を行っておいて
     ください。
     プロトコル指定の文字列は ssh,scp,plink,pscpを解釈しますが、いずれの場合も
     sshを使ってリモートファイルにアクセスしています。

   - JXRおよびBPGファイルの標準入力からの読み込みを可能にします。
     例えば「cat foo.jxr | magick convert jxr:- png:- > foo.png」はうまく動かない
     のですが、本スクリプトで一時ファイルを介する事で可能にしています。

 その他
   素の magick convertコマンドの機能を損なわないようにラップしたかったのですが、
   「+」で始まるオプション指定を解釈しないなど、Emacsから外部コマンドとして
   使用する以外の目的では不十分かも知れません。
   また、バグっていた場合は よしなに。

  wrapped_convert.d ver 2024/08/13a
 **/
import std.stdio;
import std.string;
import std.range.primitives;
import std.algorithm;
import std.process;
import std.regex;
import std.file;
import std.path;
import std.conv;
import std.uni;

enum : int {BUFSIZE = (1024*1024)}

string getelm(string[] ary, size_t p)
{
  if( p<0 || ary.length<=p )
    return null;
  return ary[p];
}

int run_convert(string[] cmd_convert)
{
  try{
    auto pipe = pipeProcess(cmd_convert, Redirect.stderr);
    scope(exit) wait(pipe.pid);
    foreach(l; pipe.stderr.byLine ){}
  }catch (Exception e){
    stderr.writeln(e);
    return 1;
  }
  return 0;
}

void run_childprocess(string[] cmd)
{
  auto pipe = pipeProcess(cmd, Redirect.stdout);
  scope(exit) wait(pipe.pid);
  foreach(line; pipe.stdout.byLine){}
  return;
}

void check_command(string cmd)
{
  auto pipe=pipeProcess([cmd], Redirect.stdout|Redirect.stderr);
  scope(exit) wait(pipe.pid);
  foreach(l; pipe.stdout.byLine ){}
  return;
}

int main(string[] args)
{
  string[] cmd_convert = ["magick", "convert"];
  string[] cmd_wconvert = [args[0]];
  string jxrdec = "JxrDecApp";
  string bpgdec = "bpgdec";

  int[string] convert_opt = [
    "-adjoin"                   : 0,
    "-affine"                   : 1,
    "-alpha"                    : 1,
    "-antialias"                : 0,
    "-authenticate"             : 1,
    "-attenuate"                : 1,
    "-background"               : 1,
    "-bias"                     : 1,
    "-black-point-compensation" : 0,
    "-blue-primary"             : 1,
    "-bordercolor"              : 1,
    "-caption"                  : 1,
    "-clip"                     : 0,
    "-clip-mask"                : 1,
    "-clip-path"                : 1,
    "-colorspace"               : 1,
    "-comment"                  : 1,
    "-compose"                  : 1,
    "-compress"                 : 1,
    "-define"                   : 1,
    "-delay"                    : 1,
    "-density"                  : 1,
    "-depth"                    : 1,
    "-direction"                : 1,
    "-display"                  : 1,
    "-dispose"                  : 1,
    "-dither"                   : 1,
    "-encoding"                 : 1,
    "-endian"                   : 1,
    "-family"                   : 1,
    "-features"                 : 1,
    "-fill"                     : 1,
    "-filter"                   : 1,
    "-font"                     : 1,
    "-format"                   : 1,
    "-fuzz"                     : 1,
    "-gravity"                  : 1,
    "-green-primary"            : 1,
    "-intensity"                : 1,
    "-intent"                   : 1,
    "-interlace"                : 1,
    "-interline-spacing"        : 1,
    "-interpolate"              : 1,
    "-interword-spacing"        : 1,
    "-kerning"                  : 1,
    "-label"                    : 1,
    "-limit"                    : 2,
    "-loop"                     : 1,
    "-matte"                    : 0,
    "-mattecolor"               : 1,
    "-moments"                  : 0,
    "-monitor"                  : 0,
    "-orient"                   : 1,
    "-page"                     : 1,
    "-ping"                     : 0,
    "-pointsize"                : 1,
    "-precision"                : 1,
    "-preview"                  : 1,
    "-quality"                  : 1,
    "-quiet"                    : 0,
    "-read-mask"                : 1,
    "-red-primary"              : 1,
    "-regard-warnings"          : 0,
    "-remap"                    : 1,
    "-repage"                   : 1,
    "-respect-parentheses"      : 0,
    "-sampling-factor"          : 1,
    "-scene"                    : 1,
    "-seed"                     : 1,
    "-size"                     : 1,
    "-stretch"                  : 1,
    "-stroke"                   : 1,
    "-strokewidth"              : 1,
    "-style"                    : 1,
    "-support"                  : 1,
    "-synchronize"              : 0,
    "-taint"                    : 0,
    "-texture"                  : 1,
    "-tile-offset"              : 1,
    "-treedepth"                : 1,
    "-transparent-color"        : 1,
    "-undercolor"               : 1,
    "-units"                    : 1,
    "-verbose"                  : 0,
    "-view"                     : 0,
    "-virtual-pixel"            : 1,
    "-weight"                   : 1,
    "-white-point"              : 1,
    "-write-mask"               : 1,
    "-adaptive-blur"            : 1,
    "-adaptive-resize"          : 1,
    "-adaptive-sharpen"         : 1,
    "-alpha"                    : 1,
    "-annotate"                 : 2,
    "-auto-gamma"               : 0,
    "-auto-level"               : 0,
    "-auto-orient"              : 0,
    "-auto-threshold"           : 1,
    "-bench"                    : 1,
    "-black-threshold"          : 1,
    "-blue-shift"               : 1,
    "-blur"                     : 1,
    "-border"                   : 1,
    "-bordercolor"              : 1,
    "-brightness-contrast"      : 0,
    "-canny"                    : 1,
    "-cdl"                      : 1,
    "-channel"                  : 1,
    "-charcoal"                 : 1,
    "-chop"                     : 1,
    "-clahe"                    : 1,
    "-clamp"                    : 0,
    "-colorize"                 : 1,
    "-color-matrix"             : 1,
    "-colors"                   : 1,
    "-connected-components"     : 1,
    "-contrast"                 : 0,
    "-contrast-stretch"         : 1,
    "-convolve"                 : 1,
    "-cycle"                    : 1,
    "-decipher"                 : 1,
    "-deskew"                   : 1,
    "-despeckle"                : 0,
    "-distort"                  : 2,
    "-draw"                     : 1,
    "-edge"                     : 1,
    "-encipher"                 : 1,
    "-emboss"                   : 1,
    "-enhance"                  : 0,
    "-equalize"                 : 0,
    "-evaluate"                 : 2,
    "-extent"                   : 1,
    "-extract"                  : 1,
    "-fft"                      : 0,
    "-flip"                     : 0,
    "-floodfill"                : 2,
    "-flop"                     : 0,
    "-frame"                    : 1,
    "-function"                 : 2,
    "-gamma"                    : 1,
    "-gaussian-blur"            : 1,
    "-geometry"                 : 1,
    "-grayscale"                : 1,
    "-hough-lines"              : 1,
    "-identify"                 : 0,
    "-ift"                      : 0,
    "-implode"                  : 1,
    "-kmeans"                   : 1,
    "-kuwahara"                 : 1,
    "-lat"                      : 1,
    "-level"                    : 1,
    "-level-colors"             : 1,
    "-linear-stretch"           : 1,
    "-liquid-rescale"           : 1,
    "-local-contrast"           : 1,
    "-mean-shift"               : 1,
    "-median"                   : 1,
    "-mode"                     : 1,
    "-modulate"                 : 1,
    "-monochrome"               : 1,
    "-morphology"               : 2,
    "-motion-blur"              : 1,
    "-negate"                   : 0,
    "-noise"                    : 1,
    "-normalize"                : 0,
    "-opaque"                   : 1,
    "-ordered-dither"           : 1,
    "-paint"                    : 1,
    "-perceptible"              : 1,
    "-polaroid"                 : 1,
    "-posterize"                : 1,
    "-profile"                  : 1,
    "-quantize"                 : 1,
    "-raise"                    : 1,
    "-random-threshold"         : 1,
    "-range-threshold"          : 1,
    "-region"                   : 1,
    "-render"                   : 0,
    "-resample"                 : 1,
    "-resize"                   : 1,
    "-roll"                     : 1,
    "-rotate"                   : 1,
    "-rotational-blur"          : 1,
    "-sample"                   : 1,
    "-scale"                    : 1,
    "-segment"                  : 1,
    "-selective-blur"           : 1,
    "-sepia-tone"               : 1,
    "-set"                      : 2,
    "-shade"                    : 1,
    "-shadow"                   : 1,
    "-sharpen"                  : 1,
    "-shave"                    : 1,
    "-shear"                    : 1,
    "-sigmoidal-contrast"       : 1,
    "-sketch"                   : 1,
    "-solarize"                 : 1,
    "-sparse-color"             : 2,
    "-splice"                   : 1,
    "-spread"                   : 1,
    "-statistic"                : 2,
    "-strip"                    : 0,
    "-swirl"                    : 1,
    "-threshold"                : 1,
    "-thumbnail"                : 1,
    "-tile"                     : 1,
    "-tint"                     : 1,
    "-transform"                : 0,
    "-transparent"              : 1,
    "-transpose"                : 0,
    "-transverse"               : 0,
    "-trim"                     : 0,
    "-type"                     : 1,
    "-unique-colors"            : 0,
    "-unsharp"                  : 1,
    "-vignette"                 : 1,
    "-wave"                     : 1,
    "-wavelet-denoise"          : 1,
    "-white-balance"            : 0,
    "-white-threshold"          : 1,
    "-channel-fx"               : 1,
    "-separate"                 : 0,
    "-append"                   : 0,
    "-clut"                     : 0,
    "-coalesce"                 : 0,
    "-combine"                  : 0,
    "-compare"                  : 0,
    "-complex"                  : 1,
    "-composite"                : 0,
    "-copy"                     : 2,
    "-crop"                     : 1,
    "-deconstruct"              : 0,
    "-evaluate-sequence"        : 1,
    "-flatten"                  : 0,
    "-fx"                       : 1,
    "-hald-clut"                : 0,
    "-layers"                   : 1,
    "-morph"                    : 1,
    "-mosaic"                   : 0,
    "-poly"                     : 1,
    "-print"                    : 1,
    "-process"                  : 1,
    "-smush"                    : 1,
    "-write"                    : 1,
    "-clone"                    : 1,
    "-delete"                   : 1,
    "-duplicate"                : 1,
    "-insert"                   : 1,
    "-reverse"                  : 0,
    "-swap"                     : 1,
    "-debug"                    : 1,
    "-distribute-cache"         : 1,
    "-help"                     : 0,
    "-list"                     : 1,
    "-log"                      : 1,
    "-version"                  : 0
  ];

  args.popFront();

  string argstr = args.joiner(" ").to!string;
  if((matchFirst(argstr, regex(`-list format`))).empty==false){
    try{
      auto pipe = pipeProcess(cmd_convert ~ args, Redirect.stdout);
      scope(exit) wait(pipe.pid);
      string[] listformat;
      foreach(line; pipe.stdout.byLine){
        listformat ~= line.idup;
      }
      foreach(line; listformat){
        writeln(line);
        if((matchFirst(line, regex(`YUV\* YUV`))).empty==false){
          bool av_jxrdec=false;
          try{
            check_command(jxrdec);
            av_jxrdec=true;
          }catch (Exception e){
          }
          writefln("      JXR  JXR       %s--   JXR File Format Syntax",
                   av_jxrdec==true ? "r" : "-");

          bool av_bpgdec=false;
          try{
            check_command(bpgdec);
            av_bpgdec=true;
          }catch (Exception e){
          }
          writefln("      BPG  BPG       %s--   BPG File Format Syntax",
                   av_bpgdec==true ? "r" : "-");
        }
      }
    }catch (Exception e){
      stderr.writeln(e);
      return 1;
    }
    return 0;
  }

  string[] files;
  for(int i=0; i<args.length; i++){
    if(args[i] in convert_opt){
      i += convert_opt[args[i]];
    }else{
      files ~= args[i];
    }
  }

  auto file0 = getelm(files, 0);
  if(file0!=null){
    auto rfile = matchFirst(file0,regex(`^\/(ssh|scp|plink|pscp):(\S+):([\s\S]+)$`));
    if(rfile.empty==false){
      string userhost = rfile[2];
      string file = rfile[3];
      string frame = "";
      auto rframe =matchFirst(file,regex(`(\[-?\d+\])$`));
      if(rframe.empty==false){
        frame = rframe[1];
      }
      file = replace(file, regex(`\[-?\d+\]$`), "");
      string ext = extension(file);
      if(sicmp(ext, ".jfif")==0){
        ext = ".jpg";
      }
      auto tmpout = std.file.deleteme() ~ ext;
      scope(exit){if(exists(tmpout)) tmpout.remove();}
      try{
        auto p = pipeProcess(["ssh", userhost, format("cat \'%s\'",file)],
                             Redirect.stdin|Redirect.stdout|Redirect.stderr);
        p.stdin.flush();
        p.stdin.close();
        while(1){
          auto buf = p.stdout.rawRead(new ubyte[BUFSIZE]);
          if(buf.length==0) break;
          tmpout.append(buf);
        }
      }catch (Exception e){
        return 1;
      }
      if(!exists(tmpout)) return 1;
      foreach(ref arg; args){
        if(arg==rfile[0]){arg=tmpout~frame; break;}
      }
      cmd_wconvert ~= args;
      return run_convert(cmd_wconvert);
    }

    auto mstdin = matchFirst(file0,regex(`^(bpg|jxr):-(\[-?\d+\])?$`,"i"));
    if(mstdin.empty==false){
      auto tmpin = std.file.deleteme() ~ "." ~ mstdin[1];
      scope(exit){if(exists(tmpin)) tmpin.remove();}

      while(1){
        auto buf=stdin.rawRead(new ubyte[BUFSIZE]);
        if(buf.length==0) break;
        tmpin.append(buf);
      }
      string tmpout;
      if(icmp(mstdin[1],"jxr")==0){
        tmpout = std.file.deleteme() ~ "." ~ "tif";
        try{
          run_childprocess([jxrdec, "-i", tmpin, "-o", tmpout]);
        }catch (Exception e){
          return 1;
        }
      }else if(icmp(mstdin[1],"bpg")==0){
        tmpout = std.file.deleteme() ~ "." ~ "png";
        try{
          run_childprocess([bpgdec, "-o", tmpout, tmpin]);
        }catch (Exception e){
          return 1;
        }
      }else{
        return 1;
      }
      scope(exit){if(exists(tmpout)) tmpout.remove();}
      foreach(ref arg; args){
        if(arg==mstdin[0]){arg=tmpout; break;}
      }
      cmd_convert ~= args;
      return run_convert(cmd_convert);
    }

    auto lfile = matchFirst(file0,regex(`\.(bpg|jxr)(\[-?\d+\])?$`,"i"));
    if(lfile.empty==false){
      auto file = replace(file0, regex(`\[-?\d+\]$`), "");
      string tmpout;
      if(icmp(lfile[1],"jxr")==0){
        tmpout = std.file.deleteme() ~ "." ~ "ppm";
        try{
          run_childprocess([jxrdec, "-i", file, "-o", tmpout]);
        }catch (Exception e){
          return 1;
        }
      }else if(icmp(lfile[1],"bpg")==0){
        tmpout = std.file.deleteme() ~ "." ~ "png";
        try{
          run_childprocess([bpgdec, "-o", tmpout, file]);
        }catch (Exception e){
          return 1;
        }
      }else{
        return 1;
      }
      scope(exit){if(exists(tmpout)) tmpout.remove();}
      foreach(ref arg; args){
        if(arg==file0){arg=tmpout; break;}
      }
      cmd_convert ~= args;
      return run_convert(cmd_convert);
    }
  }

  cmd_convert ~= args;
  return run_convert(cmd_convert);
}
